/* 
	I2C drivers, 
	by Mauro Grassi, 
	September 2010.
	
*/

#include "HardwareProfile.h"
#include "demo_board_hardware.h"
#include "hardwarestate.h"
#include "vm.h"
#include "main.h"
#include "i2c.h"

#pragma udata i2cdata

unsigned char I2CBuffer[I2C_BUFFER_SIZE];

#pragma code usercode

unsigned int getI2C(VIRTUAL_MACHINE* vm, unsigned char address, unsigned char* data, unsigned int dataLengthMax)
{
	/* 
		reads a number of bytes (up to dataLengthMax) from the slave at address=address, 
		returns the number of bytes read!	 
	*/
	unsigned int  dataLength;

	if(vm)
	{
		autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, &vm->hardwareState, OPEN_I2C_PORT);
	}			
	dataLength=0;
	/* generate a start condition */
	PIR1bits.SSP1IF=0;
	SSP1CON2bits.SEN=1;
	/* now SSP1IF is set */
	while(PIR1bits.SSP1IF==0);
	PIR1bits.SSP1IF=0;
	/* load the 7 bit address and Read Bit */
	SSP1BUF=(unsigned char)(0x01 | address);
	/* wait till SSP1IF is set */
	while(PIR1bits.SSP1IF==0);
	if(SSP1CON2bits.ACKSTAT==1)
	{
		/* ACK was not received from slave, abort */
		return dataLength;
	}
	while(dataLength<dataLengthMax)
	{
		PIR1bits.SSP1IF=0;
		/* enable receive! */
		SSP1CON2bits.RCEN=1;
		while(PIR1bits.SSP1IF==0);
		/* read data */
		*data++=SSP1BUF;
		/* send ACK signal to slave */
		if(dataLength==(dataLengthMax-1))SSP1CON2bits.ACKDT=1; else SSP1CON2bits.ACKDT=0;
		PIR1bits.SSP1IF=0;
		SSP1CON2bits.ACKEN=1;
		while(PIR1bits.SSP1IF==0);
		dataLength++;
	}
	/* generate a stop condition */	
	PIR1bits.SSP1IF=0;
	SSP1CON2bits.PEN=1;
	while(PIR1bits.SSP1IF==0);
	return dataLength;
}

unsigned int putI2C(VIRTUAL_MACHINE* vm, unsigned char address, unsigned char* data, unsigned int dataLengthMax)
{
	unsigned int written;
	
	if(vm)
	{
		autoConfigureState(vm, &thisUDL.dataLogger.ven.systemState, &vm->hardwareState, OPEN_I2C_PORT);
	}			
	/* returns the number of bytes written */
	
	/* generate a start condition */
	written=0;
	PIR1bits.SSP1IF=0;
	SSP1CON2bits.SEN=1;
	/* now SSP1IF is set */
	while(PIR1bits.SSP1IF==0);
	PIR1bits.SSP1IF=0;
	/* load the 7 bit address */
	SSP1BUF=(unsigned char)(0xFE & address);
	/* wait till SSP1IF is set */
	while(PIR1bits.SSP1IF==0);
	
	if(SSP1CON2bits.ACKSTAT==1)
	{
		/* ACK was not received from slave, abort */
		return written;
	}
	
	while(dataLengthMax--)
	{
	PIR1bits.SSP1IF=0;
	/* load the data */
	SSP1BUF=(unsigned char)*data++;
	/* wait till SSP1IF is set */
	while(PIR1bits.SSP1IF==0);
	
	if(SSP1CON2bits.ACKSTAT==1)
	{
		/* ACK was not received from slave, abort */
		return written;
	}
	written++;
	}
	/* generate a stop condition */	
	PIR1bits.SSP1IF=0;
	SSP1CON2bits.PEN=1;
	while(PIR1bits.SSP1IF==0);
	return written;
}

unsigned int putI2CTwoBytes(VIRTUAL_MACHINE* vm, unsigned char address, unsigned char data1, unsigned char data2)
{
	/* sends two bytes, returns 1 if ok, 0 otherwise */
	unsigned char* ptr;
	
	if(vm)
	{
		ptr=&vm->hardwareState.i2cPort.I2CBuffer[0];
	}
	else
	{
		ptr=&I2CBuffer[0];
	}	
	*ptr++=data1;
	*ptr++=data2;
	if(putI2C(vm, address, ptr, 2)==2)return 1; else return 0;
}

unsigned int putI2COneByte(VIRTUAL_MACHINE* vm, unsigned char address, unsigned char data1)
{
	/* sends one bytes, returns 1 if ok, 0 otherwise */
	unsigned char* ptr;
	
	if(vm)
	{
		ptr=&vm->hardwareState.i2cPort.I2CBuffer[0];
	}
	else
	{
		ptr=&I2CBuffer[0];
	}	
	*ptr++=data1;
	return putI2C(vm, address, ptr, 1);
}

unsigned int getI2CTwoBytes(VIRTUAL_MACHINE* vm, unsigned char address)
{
	/* gets two bytes, returns 1 if ok, 0 otherwise */
	unsigned char* ptr;
	
	if(vm)
	{
		ptr=&vm->hardwareState.i2cPort.I2CBuffer[0];
	}
	else
	{
		ptr=&I2CBuffer[0];
	}	
	if(getI2C(vm, address, ptr, 2)==2)return 1; else return 0;
}

unsigned int getI2COneByte(VIRTUAL_MACHINE* vm, unsigned char address)
{
	/* gets one bytes, returns 1 if ok, 0 otherwise */
	unsigned char* ptr;
	
	if(vm)
	{
		ptr=&vm->hardwareState.i2cPort.I2CBuffer[0];
	}
	else
	{
		ptr=&I2CBuffer[0];
	}	
	return getI2C(vm, address, ptr, 1);
}

void closeI2CState(I2C_DESCRIPTOR* systemState)
{
	/* closes the hardware resources */
	if(systemState->busRate==0)
	{
	
	}
	else
	{
		SSP1CON1=0;
		SSP1STAT=0;
		#if(USE_PMDIS)
			PMDIS0bits.SPI1MD=1;
		#endif	
	}
}	

void initI2CState(I2C_DESCRIPTOR* systemState)
{
	/*	
	
		Configure the MSSP1 (Master Synchronous Serial Port) module for I2C descriptor...

	*/
	unsigned char tmp;

	if(systemState->busRate==0)
	{
	
	}
	else
	{
		#if(USE_PMDIS)
			PMDIS0bits.SPI1MD=0;
		#endif	
		SSP1CON1=0x08;
		I2C_TRIS_SCL1=1;
		I2C_TRIS_SDA1=1;
		SSP1STAT=0x00;
		SSP1STATbits.SMP=0; 	/* slew rate control */
		/* note that the SSPM bits in SSP1CON1 should not be set to 1001 here because then accessing SSP1ADD actually accesses the SSPxMSK register */
		/* the default is 400 kHz */
		tmp=((((GetSystemClock()/1000)/systemState->busRate)/4)-1);		/* Select Speed */
		SSP1ADD=tmp;
		SSP1CON2=0x00;
		PIR1bits.SSP1IF=0;
		SSP1CON1bits.SSPEN=1;
		/* generate Start condition */
	}
}
